<?php
/**
 * Bigger functions and definitions
 *
 * @link https://developer.wordpress.org/themes/basics/theme-functions/
 *
 * @package Bigger
 */

if ( ! function_exists( 'bigger_setup' ) ) :
	/**
	 * Sets up theme defaults and registers support for various WordPress features.
	 *
	 * Note that this function is hooked into the after_setup_theme hook, which
	 * runs before the init hook. The init hook is too late for some features, such
	 * as indicating support for post thumbnails.
	 */
	function bigger_setup() {
		/*
		 * Make theme available for translation.
		 * Translations can be filed in the /languages/ directory.
		 * If you're building a theme based on _s, use a find and replace
		 * to change 'bigger' to the name of your theme in all the template files.
		 */
		load_theme_textdomain( 'bigger', get_template_directory() . '/languages' );

		// Add default posts and comments RSS feed links to head.
		add_theme_support( 'automatic-feed-links' );

		/*
		 * Let WordPress manage the document title.
		 * By adding theme support, we declare that this theme does not use a
		 * hard-coded <title> tag in the document head, and expect WordPress to
		 * provide it for us.
		 */
		add_theme_support( 'title-tag' );

		/*
		 * Enable support for Post Thumbnails on posts and pages.
		 *
		 * @link https://developer.wordpress.org/themes/functionality/featured-images-post-thumbnails/
		 */
		add_theme_support( 'post-thumbnails' );

		// Add image sizes
        add_image_size( 'bigger-recent-post-thumbnail', 66, 66, array( 'center', 'center' ) );

		// This theme uses wp_nav_menu() in one location.
		register_nav_menus( array(
			'primary' => esc_html__( 'Primary', 'bigger' ),
			'footer'  => esc_html__( 'Footer', 'bigger' ),
		) );

		/*
		 * Switch default core markup for search form, comment form, and comments
		 * to output valid HTML5.
		 */
		add_theme_support( 'html5', array(
			'search-form',
			'comment-form',
			'comment-list',
			'gallery',
			'caption',
		) );

		/*
		 * This theme styles the visual editor to resemble the theme style,
		 * specifically font, colors, and column width.
	 	 */
		add_editor_style( array( 'css/editor-style.css', bigger_fonts_url() ) );
		
	}
endif;
add_action( 'after_setup_theme', 'bigger_setup' );

/**
 * Set the content width in pixels, based on the theme's design and stylesheet.
 *
 * Priority 0 to make it available to lower priority callbacks.
 *
 * @global int $content_width
 */
function bigger_content_width() {
	// This variable is intended to be overruled from themes.
	// Open WPCS issue: {@link https://github.com/WordPress-Coding-Standards/WordPress-Coding-Standards/issues/1043}.
	// phpcs:ignore WordPress.NamingConventions.PrefixAllGlobals.NonPrefixedVariableFound
	$GLOBALS['content_width'] = apply_filters( 'bigger_content_width', 640 );
}
add_action( 'after_setup_theme', 'bigger_content_width', 0 );

/**
 * Register widget area.
 *
 * @link https://developer.wordpress.org/themes/functionality/sidebars/#registering-a-sidebar
 */
function bigger_widgets_init() {
	/* Register the 'primary' sidebar. */
	register_sidebar( array(
		'name'          => esc_html__( 'Primary Sidebar', 'bigger' ),
		'id'            => 'primary',
		'description'   => esc_html__( 'Add widgets here.', 'bigger' ),
		'before_widget' => '<section id="%1$s" class="widget %2$s">',
		'after_widget'  => '</section>',
		'before_title'  => '<h6 class="widget-title">',
		'after_title'   => '</h6>',
	) );

	/* Register the 'service' sidebar. */
	register_sidebar( array(
		'name'          => esc_html__( 'Service Sidebar', 'bigger' ),
		'id'            => 'service',
		'description'   => esc_html__( 'Add widgets here.', 'bigger' ),
		'before_widget' => '<div id="%1$s" class="widget %2$s">',
		'after_widget'  => '</div>',
		'before_title'  => '<h6 class="widget-title">',
		'after_title'   => '</h6>',
	) );

	/* Repeat register_sidebar() code for additional sidebars. */
	register_sidebar( array(
		'name'          => __( 'Footer First Widget Area', 'bigger' ),
		'id'            => 'footer-area-1',
		'description'   => __( 'Add widgets here to appear in your footer.', 'bigger' ),
		'before_widget' => '<div id="%1$s" class="widget %2$s">',
		'after_widget'  => '</div>',
		'before_title'  => '<h4 class="widget-title">',
		'after_title'   => '</h4>',
	) );
	register_sidebar( array(
		'name'          => __( 'Footer Second Widget Area', 'bigger' ),
		'id'            => 'footer-area-2',
		'description'   => __( 'Add widgets here to appear in your footer.', 'bigger' ),
		'before_widget' => '<div id="%1$s" class="widget %2$s">',
		'after_widget'  => '</div>',
		'before_title'  => '<h4 class="widget-title">',
		'after_title'   => '</h4>',
	) );
	register_sidebar( array(
		'name'          => __( 'Footer Third Widget Area', 'bigger' ),
		'id'            => 'footer-area-3',
		'description'   => __( 'Add widgets here to appear in your footer.', 'bigger' ),
		'before_widget' => '<div id="%1$s" class="widget %2$s">',
		'after_widget'  => '</div>',
		'before_title'  => '<h4 class="widget-title">',
		'after_title'   => '</h4>',
	) );
	register_sidebar( array(
		'name'          => __( 'Footer Fourth Widget Area', 'bigger' ),
		'id'            => 'footer-area-4',
		'description'   => __( 'Add widgets here to appear in your footer.', 'bigger' ),
		'before_widget' => '<div id="%1$s" class="widget %2$s">',
		'after_widget'  => '</div>',
		'before_title'  => '<h4 class="widget-title">',
		'after_title'   => '</h4>',
	) );
}
add_action( 'widgets_init', 'bigger_widgets_init' );

/**
 * Register custom fonts.
 */
if ( ! function_exists( 'bigger_fonts_url' ) ) :
/**
 * Register Google fonts for Blessing.
 *
 * Create your own bigger_fonts_url() function to override in a child theme.
 *
 * @since Blessing 1.0
 *
 * @return string Google fonts URL for the theme.
 */
function bigger_fonts_url() {
	$protocol 		= is_ssl() ? 'https' : 'http';
	$fonts_url 		= '';
	$font_families  = array();
	$subsets   		= 'latin,latin-ext';

	$rub  = _x( 'on', 'Rubik font: on or off', 'bigger' );
    $popp = _x( 'on', 'Poppins font: on or off', 'bigger' );

	/* translators: If there are characters in your language that are not supported by Roboto Slab, translate this to 'off'. Do not translate into your own language. */
	if ( 'off' !== $rub ) {
		$font_families[] = 'Rubik:300,300i,400,400i,500,500i,700,700i,900,900i';
	} 
	if ( 'off' !== $popp ) {
		$font_families[] = 'Poppins:100,100i,200,200i,300,300i,400,400i,500,500i,600,600i,700,700i,800,800i,900,900i';
	}

	if ( $font_families ) {
		$fonts_url = add_query_arg( array(
			'family' => urlencode( implode( '|', $font_families ) ),
			'subset' => urlencode( $subsets ),
		), 'https://fonts.googleapis.com/css' );
	}
	return esc_url_raw( $fonts_url );
}
endif;

/**
 * Enqueue scripts and styles.
 */
function bigger_scripts() {
 
	// Add custom fonts, used in the main stylesheet.
	wp_enqueue_style( 'bigger-fonts', bigger_fonts_url(), array(), null );

	/** All frontend css files **/ 
	wp_enqueue_style( 'bootstrap', get_template_directory_uri() . '/css/bootstrap.css', array(), '4.0', 'all');

	/** load fonts **/
    wp_enqueue_style( 'bigger-awesome-font', get_template_directory_uri().'/css/fontawesome.css');
    wp_enqueue_style( 'bigger-stroke-gap-icon', get_template_directory_uri().'/css/stroke-gap-icons.css');
    wp_enqueue_style( 'ionicon-font', get_template_directory_uri().'/css/ionicon.css');

    /** Slick slider **/
    wp_enqueue_style( 'slick-slider', get_template_directory_uri().'/css/slick.css');
    wp_enqueue_style( 'slick-theme', get_template_directory_uri().'/css/slick-theme.css');

    if( bigger_get_option('preload') != false ){
		wp_enqueue_style( 'bigger-preload', get_template_directory_uri().'/css/royal-preload.css');
	}

	/** Theme stylesheet. **/
	wp_enqueue_style( 'bigger-style', get_stylesheet_uri() );	

	/** Theme scripts **/
	if( bigger_get_option('preload') != false ){
		wp_enqueue_script("bigger-royal-preloader", get_template_directory_uri()."/js/royal_preloader.min.js",array('jquery'), '1.0', false); 
	}

	wp_enqueue_script( 'isotope', get_template_directory_uri() . '/js/jquery.isotope.min.js', array( 'jquery' ), '20180910', true );
	wp_enqueue_script( 'slick-slider', get_template_directory_uri() . '/js/slick.min.js', array( 'jquery' ), '20190221', true );
	wp_enqueue_script( 'elementor-widget', get_template_directory_uri() . '/js/elementor.js', array( 'jquery' ), '20190221', true );
	wp_enqueue_script( 'bigger-scripts', get_template_directory_uri() . '/js/scripts.js', array( 'jquery' ), '20180910', true );
	wp_enqueue_script( 'header-mobile-scripts', get_template_directory_uri() . '/js/header-mobile.js', array('jquery'), '20190221', true );
	wp_enqueue_script( 'jquery-masonry' );


	if ( is_singular() && comments_open() && get_option( 'thread_comments' ) ) {
		wp_enqueue_script( 'comment-reply' );
	}
}

add_action( 'wp_enqueue_scripts', 'bigger_scripts' );

/**
 * Enable by default elementor for custom post type
 */
function bigger_add_cpt_support() {
	//if exists, assign to $cpt_support var
	$cpt_support = get_option( 'elementor_cpt_support' );
	//check if option DOESN'T exist in db
	if( ! $cpt_support ) {
	    $cpt_support = [ 'page', 'post', 'ot_portfolio', 'ot_service' ]; //create array of our default supported post types
	    update_option( 'elementor_cpt_support', $cpt_support ); //write it to the database
	}
	//if it DOES exist, but ot_portfolio & ot_service is NOT defined
	else {
	    $ot_portfolio = in_array( 'ot_portfolio', $cpt_support );
		$ot_service = in_array( 'ot_service', $cpt_support );
		if( !$ot_portfolio ){
			$cpt_support[] = 'ot_portfolio'; //append to array
		}
		if( !$ot_service ){
			$cpt_support[] = 'ot_service'; //append to array
		}
		update_option( 'elementor_cpt_support', $cpt_support ); //update database
	}
	//otherwise do nothing, portfolio already exists in elementor_cpt_support option
}
add_action( 'after_switch_theme', 'bigger_add_cpt_support' );

/**
 * Custom template tags for this theme.
 */
require get_template_directory() . '/inc/frontend/template-tags.php';

/**
 * Functions which enhance the theme by hooking into WordPress.
 */
require get_template_directory() . '/inc/frontend/template-functions.php';

/**
 * Custom Page Header for this theme.
 */
require get_template_directory() . '/inc/frontend/breadcrumbs.php';
require get_template_directory() . '/inc/frontend/page-header.php';

/**
 * Functions which add more to backend.
 */
require get_template_directory() . '/inc/backend/admin-functions.php';

/**
 * Custom metabox for this theme.
 */
require get_template_directory() . '/inc/backend/meta-boxes.php';

/**
 * Customizer additions.
 */
require get_template_directory() . '/inc/backend/customizer.php';
require get_template_directory() . '/inc/backend/color.php';

/**
 * Register the required plugins for this theme.
 */
require get_template_directory() . '/inc/backend/plugin-requires.php';

require get_template_directory() . '/inc/backend/importer.php';

/**
 * Load WooCommerce compatibility file.
 */
if ( class_exists( 'WooCommerce' ) ) {
	require get_template_directory() . '/inc/woocommerce/woocommerce.php';
}

